<?php
// =======================================================
// SISTEM ABSENSI SISWA - VERSI FINAL DENGAN CRUD KELAS & SISWA
// =======================================================

session_start();
require_once 'db.php'; 

// --- KONFIGURASI ---
define('ADMIN_PASSWORD', 'Mifwan.id01');

// --- HELPER FUNCTIONS ---
function getStatusColors() {
    return [
        'H' => 'bg-green-500 text-white',
        'I' => 'bg-yellow-500 text-white',
        'S' => 'bg-blue-500 text-white',
        'A' => 'bg-red-500 text-white' 
    ];
}
function getStatusDisplayClass($status) {
    $colors = getStatusColors();
    if (isset($colors[$status])) {
        return $colors[$status];
    }
    return 'bg-gray-100 text-gray-400';
}

// --- CEK STATUS LOGIN ---
$is_admin = isset($_SESSION['is_admin_logged_in']) && $_SESSION['is_admin_logged_in'] === true;
$is_public = isset($_GET['view']) && $_GET['view'] === 'public';

// --- PROSES LOGOUT & LOGIN ---
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    session_destroy();
    header("Location: index.php");
    exit;
}
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    if ($_POST['password'] === ADMIN_PASSWORD) {
        $_SESSION['is_admin_logged_in'] = true;
        header("Location: index.php");
        exit;
    } else {
        $login_error = "Password salah!";
    }
}

// --- PROSES UPDATE KEHADIRAN (AJAX) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    if (!$is_admin) {
        echo json_encode(['success' => false, 'message' => 'Akses ditolak.']);
        exit;
    }

    // (Logika update_attendance dan get_fresh_data Anda yang asli ada di sini, tidak diubah)
    // --- LOGIKA UNTUK MENGUBAH STATUS (H, I, S, A) ---
    if ($_POST['action'] === 'update_attendance') {
        $siswa_id = $_POST['siswa_id'] ?? null;
        $tanggal = $_POST['tanggal'] ?? null;
        $status = $_POST['status'] ?? null;
        if (!$siswa_id || !$tanggal || !$status) {
            echo json_encode(['success' => false, 'message' => 'Data tidak lengkap.']);
            exit;
        }
        if ($status === 'UNMARK') {
            $query = "DELETE FROM kehadiran WHERE siswa_id = ? AND tanggal = ?";
            $stmt = $pdo->prepare($query);
            if ($stmt->execute([$siswa_id, $tanggal])) {
                echo json_encode(['success' => true, 'unmarked' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Gagal menghapus data.']);
            }
            exit;
        }
        if (in_array($status, ['H', 'I', 'S', 'A'])) {
            $query = "REPLACE INTO kehadiran (siswa_id, tanggal, status, bulan) VALUES (?, ?, ?, ?)";
            $bulan = date('Y-m', strtotime($tanggal));
            $stmt = $pdo->prepare($query);
            if ($stmt->execute([$siswa_id, $tanggal, $status, $bulan])) {
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Gagal menyimpan ke database.']);
            }
            exit;
        }
        echo json_encode(['success' => false, 'message' => 'Status tidak valid.']);
        exit;
    }

    // --- ENDPOINT AJAX UNTUK MENGAMBIL DATA TERBARU ---
    if ($_POST['action'] === 'get_fresh_data') {
        $class_id = $_POST['class_id'] ?? null;
        $month = $_POST['month'] ?? null;

        if (!$class_id || !$month) {
            echo json_encode(['success' => false, 'message' => 'Parameter kelas/bulan tidak ada.']);
            exit;
        }

        $stmt_siswa = $pdo->prepare("SELECT * FROM siswa WHERE kelas_id = ? ORDER BY nama ASC");
        $stmt_siswa->execute([$class_id]);
        $siswa_list = $stmt_siswa->fetchAll(PDO::FETCH_ASSOC);

        $attendance_data = [];
        if (!empty($siswa_list)) {
            $siswa_ids = array_column($siswa_list, 'siswa_id');
            if (!empty($siswa_ids)) {
                $placeholders = implode(',', array_fill(0, count($siswa_ids), '?'));
                $stmt_attendance = $pdo->prepare("SELECT siswa_id, tanggal, status FROM kehadiran WHERE siswa_id IN ($placeholders) AND bulan = ?");
                $stmt_attendance->execute([...$siswa_ids, $month]);
                $results = $stmt_attendance->fetchAll(PDO::FETCH_ASSOC);
                foreach ($results as $row) {
                    $attendance_data[$row['siswa_id']][$row['tanggal']] = $row['status'];
                }
            }
        }
        
        echo json_encode(['success' => true, 'attendance_data' => $attendance_data]);
        exit;
    }
}

// =======================================================
// PENAMBAHAN: PROSES ADMIN CRUD (Kelas & Siswa)
// =======================================================
if ($is_admin) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crud_action'])) {
        $crud_action = $_POST['crud_action'];

        if ($crud_action === 'add_class' && isset($_POST['nama_kelas'])) {
            $nama_kelas = trim($_POST['nama_kelas']);
            if (!empty($nama_kelas)) {
                try {
                    $stmt = $pdo->prepare("INSERT INTO kelas (nama_kelas) VALUES (?)");
                    $stmt->execute([$nama_kelas]);
                    $_SESSION['crud_message'] = ['success' => true, 'text' => "Kelas '{$nama_kelas}' berhasil ditambahkan."];
                } catch (PDOException $e) {
                    $_SESSION['crud_message'] = ['success' => false, 'text' => "Gagal menambahkan kelas: " . htmlspecialchars($e->getMessage())];
                }
            }
            header("Location: index.php?page=classes");
            exit;
        }

        if ($crud_action === 'delete_class' && isset($_POST['class_id'])) {
            $class_id = $_POST['class_id'];
            try {
                // Hapus kehadiran, siswa, lalu kelas
                $pdo->exec("DELETE FROM kehadiran WHERE siswa_id IN (SELECT siswa_id FROM siswa WHERE kelas_id = {$class_id})");
                $pdo->prepare("DELETE FROM siswa WHERE kelas_id = ?")->execute([$class_id]);
                $stmt = $pdo->prepare("DELETE FROM kelas WHERE id = ?");
                $stmt->execute([$class_id]);
                $_SESSION['crud_message'] = ['success' => true, 'text' => "Kelas dan semua data terkait berhasil dihapus."];
            } catch (PDOException $e) {
                $_SESSION['crud_message'] = ['success' => false, 'text' => "Gagal menghapus kelas: " . htmlspecialchars($e->getMessage())];
            }
            header("Location: index.php?page=classes");
            exit;
        }

        if ($crud_action === 'add_student' && isset($_POST['nama'], $_POST['kelas_id'])) {
            $nama = trim($_POST['nama']);
            $kelas_id = $_POST['kelas_id'];
            if (!empty($nama) && is_numeric($kelas_id)) {
                try {
                    // Cek duplikat siswa_id
                    $stmt = $pdo->prepare("INSERT INTO siswa (nama, kelas_id) VALUES (?, ?)");
                    $stmt->execute([$nama, $kelas_id]);
                    $new_id = $pdo->lastInsertId();
                    $_SESSION['crud_message'] = ['success' => true, 'text' => "Siswa '{$nama}' berhasil ditambahkan. ID Siswa: #{$new_id}. Harap cetak Barcode."];
                } catch (PDOException $e) {
                    $_SESSION['crud_message'] = ['success' => false, 'text' => "Gagal menambahkan siswa: " . htmlspecialchars($e->getMessage())];
                }
            }
            header("Location: index.php?page=students&class_id={$kelas_id}");
            exit;
        }

        if ($crud_action === 'delete_student' && isset($_POST['siswa_id'], $_POST['kelas_id'])) {
            $siswa_id = $_POST['siswa_id'];
            $kelas_id = $_POST['kelas_id'];
            try {
                $pdo->prepare("DELETE FROM kehadiran WHERE siswa_id = ?")->execute([$siswa_id]);
                $stmt = $pdo->prepare("DELETE FROM siswa WHERE siswa_id = ?");
                $stmt->execute([$siswa_id]);
                $_SESSION['crud_message'] = ['success' => true, 'text' => "Siswa (ID: #{$siswa_id}) berhasil dihapus."];
            } catch (PDOException $e) {
                $_SESSION['crud_message'] = ['success' => false, 'text' => "Gagal menghapus siswa: " . htmlspecialchars($e->getMessage())];
            }
            header("Location: index.php?page=students&class_id={$kelas_id}");
            exit;
        }
    }
}
$crud_message = $_SESSION['crud_message'] ?? null;
unset($_SESSION['crud_message']);
// =======================================================
// AKHIR PENAMBAHAN: PROSES ADMIN CRUD
// =======================================================


// --- AMBIL DATA DARI DATABASE (DASHBOARD) ---
$kelas_list = $pdo->query("SELECT * FROM kelas ORDER BY nama_kelas ASC")->fetchAll(PDO::FETCH_ASSOC);

// Tentukan kelas aktif untuk dashboard
$current_class_id = $_GET['class_id'] ?? ($kelas_list[0]['id'] ?? null);

$kelas_data = null;
if ($current_class_id) {
    $stmt_kelas = $pdo->prepare("SELECT * FROM kelas WHERE id = ?");
    $stmt_kelas->execute([$current_class_id]);
    $kelas_data = $stmt_kelas->fetch(PDO::FETCH_ASSOC);
}

// Tentukan bulan aktif untuk dashboard
$current_month = $_GET['month'] ?? date('Y-m');

$siswa_list = [];
if ($current_class_id) {
    $stmt_siswa = $pdo->prepare("SELECT * FROM siswa WHERE kelas_id = ? ORDER BY nama ASC");
    $stmt_siswa->execute([$current_class_id]);
    $siswa_list = $stmt_siswa->fetchAll(PDO::FETCH_ASSOC);
}

// Ambil data kehadiran untuk dashboard
$attendance_data = [];
if (!empty($siswa_list) && !empty($current_month)) {
    $siswa_ids = array_column($siswa_list, 'siswa_id');
    if (!empty($siswa_ids)) {
        $placeholders = implode(',', array_fill(0, count($siswa_ids), '?'));
        $stmt_attendance = $pdo->prepare("SELECT siswa_id, tanggal, status FROM kehadiran WHERE siswa_id IN ($placeholders) AND bulan = ?");
        $stmt_attendance->execute([...$siswa_ids, $current_month]);
        $results = $stmt_attendance->fetchAll(PDO::FETCH_ASSOC);
        foreach ($results as $row) {
            $attendance_data[$row['siswa_id']][$row['tanggal']] = $row['status'];
        }
    }
}

$days_in_month = [];
if (!empty($current_month)) {
    $date = new DateTime("$current_month-01");
    $end_date = new DateTime($date->format('Y-m-t'));
    while ($date <= $end_date) {
        $days_in_month[] = [
            'date' => $date->format('Y-m-d'),
            'day' => $date->format('j'),
            'dow' => $date->format('D'),
            'is_weekend' => in_array($date->format('D'), ['Sat', 'Sun'])
        ];
        $date->modify('+1 day');
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sistem Absensi Siswa</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .table-container { overflow: auto; max-height: 70vh; border-radius: 0.5rem; }
        .table-container thead th { position: sticky; top: 0; z-index: 10; background: #4f46e5; color: white; }
        .table-container thead th:first-child { z-index: 11; }
        .table-container tbody td:first-child { position: sticky; left: 0; z-index: 5; background: white; box-shadow: 1px 0 3px rgba(0,0,0,0.05); }
        .table-container tbody tr:hover td:first-child { background: #f3f4f6; }
        .status-btn { width: 32px; height: 32px; border-radius: 0.375rem; display: inline-flex; align-items: center; justify-content: center; font-weight: 600; font-size: 0.875rem; cursor: pointer; transition: all 0.15s ease-in-out; border: 2px solid transparent; margin: 0 1px; }
        .status-btn.active { transform: scale(1.1); border-color: #1f2937; }
        .status-btn:not(.active):hover { transform: scale(1.05); background-color: #e5e7eb; }
        .table-container::-webkit-scrollbar { height: 8px; width: 8px; }
        .table-container::-webkit-scrollbar-thumb { background-color: #cbd5e1; border-radius: 4px; }
    </style>
</head>
<body class="bg-gray-50">

<?php if (!$is_admin && !$is_public): ?>
<div class="min-h-screen flex items-center justify-center bg-gradient-to-br from-indigo-500 to-purple-600">
    <div class="bg-white p-8 rounded-2xl shadow-2xl w-full max-w-md">
        <div class="text-center mb-6">
            <div class="inline-flex items-center justify-center w-16 h-16 bg-indigo-100 rounded-full mb-4">
                <i class="fas fa-user-shield text-indigo-600 text-2xl"></i>
            </div>
            <h2 class="text-3xl font-bold text-gray-900">Login Admin</h2>
            <p class="text-gray-600">Masuk untuk mengakses dashboard</p>
        </div>
        <?php if (isset($login_error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-4 text-sm"><?= htmlspecialchars($login_error) ?></div>
        <?php endif; ?>
        <form method="POST" class="space-y-6">
            <input type="hidden" name="login" value="1">
            <div>
                <label for="password" class="block text-sm font-medium text-gray-700">Password</label>
                <input type="password" id="password" name="password" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500">
            </div>
            <button type="submit" class="w-full bg-indigo-600 text-white py-2 px-4 rounded-lg hover:bg-indigo-700 transition font-semibold">Masuk</button>
        </form>
    </div>
</div>

<?php else: ?>
<div class="flex min-h-screen bg-gray-100">
    <aside class="w-64 bg-gray-800 text-white flex-shrink-0 hidden md:block">
        <div class="p-6">
            <h1 class="text-2xl font-bold flex items-center text-indigo-400">
                <i class="fas fa-clipboard-check mr-2"></i> SDM03KREATIF
            </h1>
        </div>
        <nav class="mt-6">
            <?php $current_page = $_GET['page'] ?? 'dashboard'; ?>
            
            <a href="index.php" class="block py-3 px-6 <?= ($current_page === 'dashboard' && !isset($_GET['page'])) ? 'bg-indigo-600 text-white font-semibold border-l-4 border-white' : 'text-gray-300 hover:bg-gray-700 hover:text-white transition' ?>">Dashboard</a>
            
            <a href="scan.php" class="block py-3 px-6 text-gray-300 hover:bg-gray-700 hover:text-white transition">
                <i class="fas fa-qrcode mr-2"></i> Scan Absen
            </a>
            <a href="cetak_barcode.php" class="block py-3 px-6 text-gray-300 hover:bg-gray-700 hover:text-white transition">
                <i class="fas fa-barcode mr-2"></i> Cetak Barcode
            </a>
            
            <?php if ($is_admin): ?>
                <div class="p-6 pt-4 text-xs font-semibold text-gray-400 uppercase">Administrasi</div>
                <a href="index.php?page=classes" class="block py-3 px-6 <?= $current_page === 'classes' ? 'bg-indigo-600 text-white font-semibold border-l-4 border-white' : 'text-gray-300 hover:bg-gray-700 hover:text-white transition' ?>">
                    <i class="fas fa-chalkboard-teacher mr-2"></i> Kelola Kelas
                </a>
                <a href="index.php?page=students&class_id=<?= $current_class_id ?>" class="block py-3 px-6 <?= $current_page === 'students' ? 'bg-indigo-600 text-white font-semibold border-l-4 border-white' : 'text-gray-300 hover:bg-gray-700 hover:text-white transition' ?>">
                    <i class="fas fa-users mr-2"></i> Kelola Siswa
                </a>
            <?php endif; ?>
            
            <div class="absolute bottom-0 w-64 p-6 border-t border-gray-700">
                <?php if ($is_admin): ?>
                    <p class="text-xs text-gray-400 mb-2">MODE: ADMINISTRATOR</p>
                    <a href="?view=public" class="block py-2 px-4 text-sm hover:bg-gray-700 rounded transition">Mode Publik</a>
                    <a href="?action=logout" class="block py-2 px-4 text-sm hover:bg-gray-700 rounded transition text-red-300">Logout</a>
                <?php else: ?>
                    <p class="text-xs text-gray-400 mb-2">MODE: PUBLIK</p>
                    <a href="index.php" class="block py-2 px-4 text-sm hover:bg-gray-700 rounded transition text-indigo-300">Login Admin</a>
                <?php endif; ?>
            </div>
        </nav>
    </aside>

    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow p-4 md:p-6 sticky top-0 z-40">
            <h2 class="text-2xl font-bold text-gray-800">
                <?php 
                    if ($current_page === 'classes') echo 'Administrasi Kelas';
                    elseif ($current_page === 'students') echo 'Administrasi Siswa';
                    else echo 'Rekap Absensi Siswa';
                ?>
            </h2>
        </header>

        <div class="p-4 md:p-6">
            <?php if (isset($crud_message)): ?>
                <div class="bg-<?= $crud_message['success'] ? 'green' : 'red' ?>-100 border border-<?= $crud_message['success'] ? 'green' : 'red' ?>-400 text-<?= $crud_message['success'] ? 'green' : 'red' ?>-700 px-4 py-3 rounded-lg mb-4 text-sm">
                    <?= htmlspecialchars($crud_message['text']) ?>
                </div>
            <?php endif; ?>

            <?php 
            $page = $_GET['page'] ?? 'dashboard';
            
            if ($is_admin && $page === 'classes'): 
                // ===================================
                // HALAMAN: KELOLA KELAS
                // ===================================
            ?>
            
                <h2 class="text-3xl font-bold text-gray-800 mb-6">Kelola Kelas</h2>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div class="bg-white p-6 rounded-lg shadow col-span-1">
                        <h3 class="text-xl font-semibold mb-4">Tambah Kelas Baru</h3>
                        <form method="POST" class="space-y-4">
                            <input type="hidden" name="crud_action" value="add_class">
                            <div>
                                <label for="nama_kelas" class="block text-sm font-medium text-gray-700">Nama Kelas (Contoh: X RPL)</label>
                                <input type="text" id="nama_kelas" name="nama_kelas" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                            </div>
                            <button type="submit" class="w-full bg-indigo-600 text-white py-2 px-4 rounded-md hover:bg-indigo-700 transition font-semibold">
                                <i class="fas fa-plus mr-1"></i> Tambah Kelas
                            </button>
                        </form>
                    </div>

                    <div class="bg-white p-6 rounded-lg shadow col-span-2">
                        <h3 class="text-xl font-semibold mb-4">Daftar Kelas Aktif (<?= count($kelas_list) ?>)</h3>
                        <div class="overflow-y-auto" style="max-height: 400px;">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50 sticky top-0">
                                    <tr>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nama Kelas</th>
                                        <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Aksi</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-200">
                                    <?php foreach ($kelas_list as $k): ?>
                                        <tr>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                                <?= htmlspecialchars($k['nama_kelas']) ?>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                                                <form method="POST" onsubmit="return confirm('Apakah Anda yakin menghapus kelas <?= htmlspecialchars($k['nama_kelas']) ?>? SEMUA SISWA & DATA KEHADIRANNYA akan ikut terhapus!');" class="inline">
                                                    <input type="hidden" name="crud_action" value="delete_class">
                                                    <input type="hidden" name="class_id" value="<?= $k['id'] ?>">
                                                    <button type="submit" class="text-red-600 hover:text-red-900 text-lg" title="Hapus Kelas">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

            <?php 
            elseif ($is_admin && $page === 'students'): 
                // ===================================
                // HALAMAN: KELOLA SISWA
                // ===================================
                $current_class_for_students = $_GET['class_id'] ?? ($kelas_list[0]['id'] ?? null);

                $stmt_kelas_select = $pdo->prepare("SELECT * FROM kelas WHERE id = ?");
                $stmt_kelas_select->execute([$current_class_for_students]);
                $selected_class_data = $stmt_kelas_select->fetch(PDO::FETCH_ASSOC);

                $siswa_list_crud = [];
                if ($current_class_for_students) {
                    $stmt_siswa_select = $pdo->prepare("SELECT * FROM siswa WHERE kelas_id = ? ORDER BY nama ASC");
                    $stmt_siswa_select->execute([$current_class_for_students]);
                    $siswa_list_crud = $stmt_siswa_select->fetchAll(PDO::FETCH_ASSOC);
                }
            ?>
            
                <h2 class="text-3xl font-bold text-gray-800 mb-6">Kelola Siswa</h2>

                <div class="bg-white p-6 rounded-lg shadow mb-6">
                    <h3 class="text-xl font-semibold mb-4">Pilih Kelas untuk Dikelola</h3>
                    <form method="GET">
                        <input type="hidden" name="page" value="students">
                        <select name="class_id" onchange="this.form.submit()" class="mt-1 block w-full md:w-1/3 px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                            <?php foreach ($kelas_list as $k): ?>
                                <option value="<?= $k['id'] ?>" <?= $k['id'] == $current_class_for_students ? 'selected' : '' ?>><?= htmlspecialchars($k['nama_kelas']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </form>
                </div>

                <?php if ($selected_class_data): ?>
                    
                    <h3 class="text-2xl font-bold text-indigo-600 mb-4">Daftar Siswa di Kelas: <?= htmlspecialchars($selected_class_data['nama_kelas']) ?></h3>

                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div class="bg-white p-6 rounded-lg shadow col-span-1 h-fit">
                            <h4 class="text-xl font-semibold mb-4">Tambah Siswa Baru</h4>
                            <form method="POST" class="space-y-4">
                                <input type="hidden" name="crud_action" value="add_student">
                                <input type="hidden" name="kelas_id" value="<?= $selected_class_data['id'] ?>">
                                <div>
                                    <label for="nama_siswa" class="block text-sm font-medium text-gray-700">Nama Siswa</label>
                                    <input type="text" id="nama_siswa" name="nama" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                                </div>
                                <button type="submit" class="w-full bg-green-600 text-white py-2 px-4 rounded-md hover:bg-green-700 transition font-semibold">
                                    <i class="fas fa-user-plus mr-1"></i> Tambah Siswa
                                </button>
                                <p class="text-xs text-gray-500 mt-2">ID Siswa akan dibuat otomatis.</p>
                            </form>
                        </div>

                        <div class="bg-white p-6 rounded-lg shadow col-span-2">
                            <h4 class="text-xl font-semibold mb-4">Daftar Siswa (<?= count($siswa_list_crud) ?>)</h4>
                            <div class="overflow-y-auto" style="max-height: 500px;">
                                <table class="min-w-full divide-y divide-gray-200">
                                    <thead class="bg-gray-50 sticky top-0">
                                        <tr>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID Siswa</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nama Siswa</th>
                                            <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody class="bg-white divide-y divide-gray-200">
                                        <?php foreach ($siswa_list_crud as $s): ?>
                                            <tr>
                                                <td class="px-6 py-4 whitespace-nowrap text-sm font-bold text-gray-500">#<?= $s['siswa_id'] ?></td>
                                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                                    <?= htmlspecialchars($s['nama']) ?>
                                                </td>
                                                <td class="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                                                    <form method="POST" onsubmit="return confirm('Apakah Anda yakin menghapus siswa <?= htmlspecialchars($s['nama']) ?> (ID: #<?= $s['siswa_id'] ?>)? Data kehadirannya juga akan terhapus!');" class="inline">
                                                        <input type="hidden" name="crud_action" value="delete_student">
                                                        <input type="hidden" name="siswa_id" value="<?= $s['siswa_id'] ?>">
                                                        <input type="hidden" name="kelas_id" value="<?= $selected_class_data['id'] ?>">
                                                        <button type="submit" class="text-red-600 hover:text-red-900 text-lg" title="Hapus Siswa">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                <?php endif; ?>

            <?php else: ?>
                <div class="bg-white p-6 rounded-lg shadow mb-6">
                    <form method="GET" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <?php if ($is_public) echo '<input type="hidden" name="view" value="public">'; ?>
                        <div>
                            <label class="block text-sm font-medium text-gray-700">Kelas</label>
                            <select name="class_id" onchange="this.form.submit()" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                                <?php if (empty($kelas_list)): ?>
                                    <option disabled>Tidak ada kelas. Silakan Kelola Kelas dulu.</option>
                                <?php endif; ?>
                                <?php foreach ($kelas_list as $k): ?>
                                    <option value="<?= $k['id'] ?>" <?= $k['id'] == $current_class_id ? 'selected' : '' ?>><?= htmlspecialchars($k['nama_kelas']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700">Bulan</label>
                            <input type="month" name="month" value="<?= $current_month ?>" onchange="this.form.submit()" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500">
                        </div>
                        <div class="flex items-end">
                            <div class="bg-indigo-100 text-indigo-800 p-3 rounded-md w-full text-center">
                                <p class="text-sm">Kelas Aktif</p>
                                <p class="font-bold"><?= htmlspecialchars($kelas_data['nama_kelas'] ?? 'Pilih Kelas') ?></p>
                            </div>
                        </div>
                    </form>
                </div>

                <?php if (empty($siswa_list)): ?>
                    <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 text-center"><p class="text-yellow-700">Tidak ada siswa di kelas ini.</p></div>
                <?php else: ?>
                    <div class="bg-white rounded-lg shadow mb-6">
                        <div class="p-4 border-b font-semibold text-lg">Tabel Kehadiran Harian</div>
                        <div class="table-container shadow-inner">
                            <table class="w-full text-sm border-collapse">
                                <thead>
                                    <tr>
                                        <th class="p-3 text-left border-b font-bold tracking-wider w-40">Nama Siswa</th>
                                        <?php foreach ($days_in_month as $day): ?>
                                            <th class="p-2 text-center border-b font-bold <?= $day['is_weekend'] ? 'bg-pink-600' : '' ?>">
                                                <div class="text-lg"><?= $day['day'] ?></div>
                                                <div class="text-xs font-normal opacity-70"><?= $day['dow'] ?></div>
                                            </th>
                                        <?php endforeach; ?>
                                    </tr>
                                </thead>
                                <tbody id="attendance-table-body">
                                    <?php foreach ($siswa_list as $siswa): ?>
                                        <tr class="hover:bg-gray-50">
                                            <td class="p-3 font-medium border-b text-gray-700"><?= htmlspecialchars($siswa['nama']) ?></td>
                                            <?php foreach ($days_in_month as $day): ?>
                                                <td class="p-2 text-center border-b">
                                                    <?php $current_status = $attendance_data[$siswa['siswa_id']][$day['date']] ?? ''; $status_colors = getStatusColors(); ?>
                                                    <?php if ($is_admin): ?>
                                                        <div class="status-container inline-flex" data-siswa-id="<?= $siswa['siswa_id'] ?>" data-tanggal="<?= $day['date'] ?>" data-current-status="<?= $current_status ?>">
                                                            <?php foreach (['H', 'I', 'S', 'A'] as $status): ?>
                                                                <button class="status-btn <?= $status == $current_status ? $status_colors[$status] . ' active' : 'bg-gray-200 text-gray-600' ?>" data-status="<?= $status ?>" title="<?= $status == 'H' ? 'Hadir' : ($status == 'I' ? 'Izin' : ($status == 'S' ? 'Sakit' : 'Alpha')) ?>"><?= $status ?></button>
                                                            <?php endforeach; ?>
                                                        </div>
                                                    <?php else: ?>
                                                        <span class="inline-block px-2 py-1 text-xs font-semibold rounded-full <?= getStatusDisplayClass($current_status) ?>"><?= $current_status ?: '-' ?></span>
                                                    <?php endif; ?>
                                                </td>
                                            <?php endforeach; ?>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <div class="bg-white rounded-lg shadow mt-6">
                        <div class="p-4 border-b font-semibold text-lg">Rekapitulasi Bulanan</div>
                        <div class="p-6">
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8" id="total-cards-container"></div>
                            <div class="overflow-x-auto">
                                <table class="w-full text-sm border-collapse">
                                    <thead class="bg-gray-50 border-b">
                                        <tr>
                                            <th class="p-3 text-left font-bold text-gray-700">Nama Siswa</th>
                                            <th class="p-3 text-center font-bold text-green-600">Hadir</th>
                                            <th class="p-3 text-center font-bold text-yellow-600">Izin</th>
                                            <th class="p-3 text-center font-bold text-blue-600">Sakit</th>
                                            <th class="p-3 text-center font-bold text-red-600">Alpha</th>
                                            <th class="p-3 text-center font-bold text-indigo-600">Total Hari</th>
                                            <th class="p-3 text-center font-bold text-indigo-600">Persentase H</th>
                                        </tr>
                                    </thead>
                                    <tbody id="summary-table-body"></tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </main>
</div>

<div id="toast" class="fixed bottom-4 right-4 bg-green-500 text-white px-6 py-3 rounded-lg shadow-xl transform translate-y-full transition-transform duration-300 z-50"><span id="toast-message"></span></div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const statusColorsMap = { 'H': 'bg-green-500 text-white', 'I': 'bg-yellow-500 text-white', 'S': 'bg-blue-500 text-white', 'A': 'bg-red-500 text-white' };
    const defaultClasses = 'bg-gray-200 text-gray-600'; const activeBorderClass = 'active';
    function showToast(message, isError = false) {
        const toast = document.getElementById('toast'); const toastMessage = document.getElementById('toast-message');
        toastMessage.textContent = message; toast.className = `fixed bottom-4 right-4 px-6 py-3 rounded-lg shadow-xl transform transition-transform duration-300 z-50 ${isError ? 'bg-red-500' : 'bg-green-500'} text-white`;
        toast.classList.remove('translate-y-full'); setTimeout(() => toast.classList.add('translate-y-full'), 3000);
    }
    function updateStatusButton(button, isActive) {
        const status = button.dataset.status; button.className = 'status-btn';
        if (isActive) { button.classList.add(...statusColorsMap[status].split(' '), activeBorderClass); }
        else { button.classList.add(...defaultClasses.split(' ')); button.classList.remove(activeBorderClass); }
    }
    document.querySelectorAll('.status-container').forEach(container => {
        const initialStatus = container.dataset.currentStatus;
        container.querySelectorAll('.status-btn').forEach(btn => { updateStatusButton(btn, btn.dataset.status === initialStatus); });
    });
    const tableBody = document.getElementById('attendance-table-body');
    if (tableBody) {
        tableBody.addEventListener('click', async function(e) {
            if (e.target.classList.contains('status-btn')) {
                const button = e.target; const container = button.closest('.status-container');
                const siswaId = container.dataset.siswaId; const tanggal = container.dataset.tanggal;
                let newStatus = button.dataset.status; const currentStatus = container.dataset.currentStatus;
                
                if (!currentStatus) {
                    // Jika status kosong, klik pertama akan memilih status baru
                    updateStatusButton(button, true); 
                    container.dataset.currentStatus = newStatus;
                } else if (button.classList.contains(activeBorderClass)) { 
                    // Jika tombol aktif diklik, UNMARK (hapus status)
                    newStatus = 'UNMARK'; 
                    updateStatusButton(button, false); 
                    container.dataset.currentStatus = '';
                } else {
                    // Jika tombol lain diklik, ganti status
                    container.querySelectorAll('.status-btn.active').forEach(btn => updateStatusButton(btn, false)); 
                    updateStatusButton(button, true); 
                    container.dataset.currentStatus = newStatus;
                }

                container.style.opacity = '0.5'; container.style.pointerEvents = 'none';
                try {
                    const response = await fetch('', { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body: new URLSearchParams({ action: 'update_attendance', siswa_id: siswaId, tanggal: tanggal, status: newStatus }) });
                    const result = await response.json();
                    if (result.success) { 
                        showToast(newStatus === 'UNMARK' ? 'Status berhasil dihapus (Kosong)' : `Status berhasil diubah menjadi ${newStatus}`); 
                        updateSummary(); 
                    }
                    else { 
                        showToast(result.message || 'Gagal mengubah status.', true); 
                        // Kembalikan ke status awal jika gagal
                        container.dataset.currentStatus = currentStatus; 
                        container.querySelectorAll('.status-btn').forEach(btn => { updateStatusButton(btn, btn.dataset.status === currentStatus); }); 
                    }
                } catch (error) { 
                    console.error('Error:', error); 
                    showToast('Terjadi kesalahan jaringan.', true); 
                    // Kembalikan ke status awal jika gagal
                    container.dataset.currentStatus = currentStatus; 
                    container.querySelectorAll('.status-btn').forEach(btn => { updateStatusButton(btn, btn.dataset.status === currentStatus); }); 
                }
                finally { container.style.opacity = '1'; container.style.pointerEvents = 'auto'; }
            }
        });
    }

    // =======================================================
    // PEMBARUAN FINAL: Fungsi updateSummary yang paling andal
    // =======================================================
    async function updateSummary() {
        try {
            const response = await fetch('', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'get_fresh_data',
                    class_id: '<?= $current_class_id ?>',
                    month: '<?= $current_month ?>'
                })
            });

            const result = await response.json();

            if (result.success) {
                const freshAttendanceData = result.attendance_data;

                // 1. PERBAIKAN UTAMA: Update Tabel Kehadiran Harian berdasarkan data fresh
                document.querySelectorAll('.status-container').forEach(container => {
                    const siswaId = container.dataset.siswaId;
                    const tanggal = container.dataset.tanggal;
                    // Ambil status dari data terbaru, jika tidak ada, default ke string kosong
                    const currentStatusFromServer = freshAttendanceData[siswaId] ? (freshAttendanceData[siswaId][tanggal] ?? '') : '';
                    
                    // Update attribute data
                    container.dataset.currentStatus = currentStatusFromServer;

                    // Update tampilan tombol
                    container.querySelectorAll('.status-btn').forEach(btn => {
                        const btnStatus = btn.dataset.status;
                        btn.className = 'status-btn';
                        if (btnStatus === currentStatusFromServer) {
                            btn.classList.add(...statusColorsMap[btnStatus].split(' '), activeBorderClass);
                        } else {
                            btn.classList.add(...defaultClasses.split(' '));
                        }
                    });
                });

                // 2. Update Rekapitulasi berdasarkan data FRESH dari server
                const summaryData = {};
                const totalDays = document.querySelectorAll('thead th').length - 1; 
                document.querySelectorAll('#attendance-table-body tr').forEach(row => {
                    const nama = row.cells[0].textContent;
                    const container = row.querySelector('.status-container');
                    if (container) {
                        const siswaId = container.dataset.siswaId;
                        summaryData[siswaId] = { nama: nama, H: 0, I: 0, S: 0, A: 0, total_days: totalDays };
                        
                        if (freshAttendanceData[siswaId]) {
                            for (const date in freshAttendanceData[siswaId]) {
                                const status = freshAttendanceData[siswaId][date];
                                if (summaryData[siswaId].hasOwnProperty(status)) {
                                    summaryData[siswaId][status]++;
                                }
                            }
                        }
                    }
                });

                const totals = { H: 0, I: 0, S: 0, A: 0 };
                Object.values(summaryData).forEach(data => { totals.H += data.H; totals.I += data.I; totals.S += data.S; totals.A += data.A; });
                const cardsContainer = document.getElementById('total-cards-container');
                if (cardsContainer) {
                    cardsContainer.innerHTML = `<div class="bg-green-100 p-4 rounded-lg text-center shadow-sm"><p class="text-green-800 font-bold text-2xl">${totals.H}</p><p class="text-green-600 font-medium">Hadir</p></div><div class="bg-yellow-100 p-4 rounded-lg text-center shadow-sm"><p class="text-yellow-800 font-bold text-2xl">${totals.I}</p><p class="text-yellow-600 font-medium">Izin</p></div><div class="bg-blue-100 p-4 rounded-lg text-center shadow-sm"><p class="text-blue-800 font-bold text-2xl">${totals.S}</p><p class="text-blue-600 font-medium">Sakit</p></div><div class="bg-red-100 p-4 rounded-lg text-center shadow-sm"><p class="text-red-800 font-bold text-2xl">${totals.A}</p><p class="text-red-600 font-medium">Alpha</p></div>`;
                }
                
                const summaryTableBody = document.getElementById('summary-table-body'); 
                if (summaryTableBody) {
                    summaryTableBody.innerHTML = '';
                    for (const data of Object.values(summaryData)) {
                        const totalMarked = data.H + data.I + data.S + data.A; 
                        const percentage = totalMarked > 0 ? ((data.H / totalMarked) * 100).toFixed(1) : 0;
                        const percentageColor = percentage >= 90 ? 'text-green-600' : (percentage >= 75 ? 'text-yellow-600' : 'text-red-600');
                        const row = summaryTableBody.insertRow(); row.className = 'even:bg-gray-50';
                        row.innerHTML = `<td class="p-3 border-b text-left">${data.nama}</td><td class="p-3 text-center border-b font-semibold text-green-700">${data.H}</td><td class="p-3 text-center border-b font-semibold text-yellow-700">${data.I}</td><td class="p-3 text-center border-b font-semibold text-blue-700">${data.S}</td><td class="p-3 text-center border-b font-semibold text-red-700">${data.A}</td><td class="p-3 text-center border-b font-bold">${totalMarked} / ${totalDays}</td><td class="p-3 text-center border-b font-bold ${percentageColor}">${percentage}%</td>`;
                    }
                }

            } else {
                showToast('Gagal memperbarui data.', true);
            }
        } catch (error) {
            console.error('Error updating summary:', error);
            showToast('Terjadi kesalahan jaringan saat memperbarui data.', true);
        }
    }
    // =======================================================
    // AKHIR PEMBARUAN FINAL
    // =======================================================

    // Inisialisasi rekapitulasi saat halaman dimuat (hanya jika di dashboard)
    if (document.getElementById('total-cards-container')) {
        updateSummary();
    }
});
</script>

<?php endif; ?>
</body>
</html>